# The contents of this file are automatically written by
# tools/generate_schema_wrapper.py. Do not modify directly.

# These errors need to be ignored as they come from the overload methods
# which trigger two kind of errors in mypy:
# * all of them do not have an implementation in this file
# * some of them are the only overload methods -> overloads usually only make
#   sense if there are multiple ones
# However, we need these overloads due to how the propertysetter works
# mypy: disable-error-code="no-overload-impl, empty-body, misc"

from __future__ import annotations

from typing import TYPE_CHECKING, Any, Literal, Sequence, TypedDict, Union, overload
from typing_extensions import TypeAlias

import narwhals.stable.v1 as nw

from altair.utils import infer_encoding_types as _infer_encoding_types
from altair.utils import parse_shorthand
from altair.utils.schemapi import Undefined, with_property_setters

from . import core
from ._typing import *  # noqa: F403

# ruff: noqa: F405
if TYPE_CHECKING:
    from typing_extensions import Self

    from altair import Parameter, SchemaBase
    from altair.typing import Optional


__all__ = [
    "X2",
    "Y2",
    "Angle",
    "AngleDatum",
    "AngleValue",
    "Color",
    "ColorDatum",
    "ColorValue",
    "Column",
    "DatumChannelMixin",
    "Description",
    "DescriptionValue",
    "Detail",
    "Facet",
    "FieldChannelMixin",
    "Fill",
    "FillDatum",
    "FillOpacity",
    "FillOpacityDatum",
    "FillOpacityValue",
    "FillValue",
    "Href",
    "HrefValue",
    "Key",
    "Latitude",
    "Latitude2",
    "Latitude2Datum",
    "Latitude2Value",
    "LatitudeDatum",
    "Longitude",
    "Longitude2",
    "Longitude2Datum",
    "Longitude2Value",
    "LongitudeDatum",
    "Opacity",
    "OpacityDatum",
    "OpacityValue",
    "Order",
    "OrderValue",
    "Radius",
    "Radius2",
    "Radius2Datum",
    "Radius2Value",
    "RadiusDatum",
    "RadiusValue",
    "Row",
    "Shape",
    "ShapeDatum",
    "ShapeValue",
    "Size",
    "SizeDatum",
    "SizeValue",
    "Stroke",
    "StrokeDash",
    "StrokeDashDatum",
    "StrokeDashValue",
    "StrokeDatum",
    "StrokeOpacity",
    "StrokeOpacityDatum",
    "StrokeOpacityValue",
    "StrokeValue",
    "StrokeWidth",
    "StrokeWidthDatum",
    "StrokeWidthValue",
    "Text",
    "TextDatum",
    "TextValue",
    "Theta",
    "Theta2",
    "Theta2Datum",
    "Theta2Value",
    "ThetaDatum",
    "ThetaValue",
    "Tooltip",
    "TooltipValue",
    "Url",
    "UrlValue",
    "ValueChannelMixin",
    "X",
    "X2Datum",
    "X2Value",
    "XDatum",
    "XError",
    "XError2",
    "XError2Value",
    "XErrorValue",
    "XOffset",
    "XOffsetDatum",
    "XOffsetValue",
    "XValue",
    "Y",
    "Y2Datum",
    "Y2Value",
    "YDatum",
    "YError",
    "YError2",
    "YError2Value",
    "YErrorValue",
    "YOffset",
    "YOffsetDatum",
    "YOffsetValue",
    "YValue",
    "with_property_setters",
]


class FieldChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict | list[dict]:
        context = context or {}
        ignore = ignore or []
        shorthand = self._get("shorthand")  # type: ignore[attr-defined]
        field = self._get("field")  # type: ignore[attr-defined]

        if shorthand is not Undefined and field is not Undefined:
            msg = f"{self.__class__.__name__} specifies both shorthand={shorthand} and field={field}. "
            raise ValueError(msg)

        if isinstance(shorthand, (tuple, list)):
            # If given a list of shorthands, then transform it to a list of classes
            kwds = self._kwds.copy()  # type: ignore[attr-defined]
            kwds.pop("shorthand")
            return [
                self.__class__(sh, **kwds).to_dict(  # type: ignore[call-arg]
                    validate=validate, ignore=ignore, context=context
                )
                for sh in shorthand
            ]

        if shorthand is Undefined:
            parsed = {}
        elif isinstance(shorthand, str):
            data: nw.DataFrame | Any = context.get("data", None)
            parsed = parse_shorthand(shorthand, data=data)
            type_required = "type" in self._kwds  # type: ignore[attr-defined]
            type_in_shorthand = "type" in parsed
            type_defined_explicitly = self._get("type") is not Undefined  # type: ignore[attr-defined]
            if not type_required:
                # Secondary field names don't require a type argument in VegaLite 3+.
                # We still parse it out of the shorthand, but drop it here.
                parsed.pop("type", None)
            elif not (type_in_shorthand or type_defined_explicitly):
                if isinstance(data, nw.DataFrame):
                    msg = (
                        f'Unable to determine data type for the field "{shorthand}";'
                        " verify that the field name is not misspelled."
                        " If you are referencing a field from a transform,"
                        " also confirm that the data type is specified correctly."
                    )
                    raise ValueError(msg)
                else:
                    msg = (
                        f"{shorthand} encoding field is specified without a type; "
                        "the type cannot be automatically inferred because "
                        "the data is not specified as a pandas.DataFrame."
                    )
                    raise ValueError(msg)
        else:
            # Shorthand is not a string; we pass the definition to field,
            # and do not do any parsing.
            parsed = {"field": shorthand}
        context["parsed_shorthand"] = parsed

        return super().to_dict(validate=validate, ignore=ignore, context=context)


class ValueChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict:
        context = context or {}
        ignore = ignore or []
        condition = self._get("condition", Undefined)  # type: ignore[attr-defined]
        copy = self  # don't copy unless we need to
        if condition is not Undefined:
            if isinstance(condition, core.SchemaBase):
                pass
            elif "field" in condition and "type" not in condition:
                kwds = parse_shorthand(condition["field"], context.get("data", None))
                copy = self.copy(deep=["condition"])  # type: ignore[attr-defined]
                copy["condition"].update(kwds)  # type: ignore[index]
        return super(ValueChannelMixin, copy).to_dict(
            validate=validate, ignore=ignore, context=context
        )


class DatumChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict:
        context = context or {}
        ignore = ignore or []
        datum = self._get("datum", Undefined)  # type: ignore[attr-defined] # noqa
        copy = self  # don't copy unless we need to
        return super(DatumChannelMixin, copy).to_dict(
            validate=validate, ignore=ignore, context=context
        )


@with_property_setters
class Angle(FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber):
    r"""
    Angle schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Angle: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Angle: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Angle: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Angle: ...

    @overload
    def bin(self, _: bool, **kwds) -> Angle: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def bin(self, _: None, **kwds) -> Angle: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> Angle: ...

    @overload
    def field(self, _: str, **kwds) -> Angle: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def legend(self, _: None, **kwds) -> Angle: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def scale(self, _: None, **kwds) -> Angle: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Angle: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Angle: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Angle: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Angle: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Angle: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Angle: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Angle: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def sort(self, _: None, **kwds) -> Angle: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Angle: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Angle: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Angle: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Angle: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Angle: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Angle: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Angle: ...

    @overload
    def title(self, _: str, **kwds) -> Angle: ...

    @overload
    def title(self, _: list[str], **kwds) -> Angle: ...

    @overload
    def title(self, _: None, **kwds) -> Angle: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Angle: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class AngleDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    AngleDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def bandPosition(self, _: float, **kwds) -> AngleDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> AngleDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> AngleDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> AngleDatum: ...

    @overload
    def title(self, _: str, **kwds) -> AngleDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> AngleDatum: ...

    @overload
    def title(self, _: None, **kwds) -> AngleDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> AngleDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class AngleValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    AngleValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> AngleValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> AngleValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Color(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Color schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Color: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Color: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Color: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Color: ...

    @overload
    def bin(self, _: bool, **kwds) -> Color: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def bin(self, _: None, **kwds) -> Color: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> Color: ...

    @overload
    def field(self, _: str, **kwds) -> Color: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def legend(self, _: None, **kwds) -> Color: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def scale(self, _: None, **kwds) -> Color: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Color: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Color: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Color: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Color: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Color: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Color: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Color: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def sort(self, _: None, **kwds) -> Color: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Color: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Color: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Color: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Color: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Color: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Color: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Color: ...

    @overload
    def title(self, _: str, **kwds) -> Color: ...

    @overload
    def title(self, _: list[str], **kwds) -> Color: ...

    @overload
    def title(self, _: None, **kwds) -> Color: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Color: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ColorDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    ColorDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def bandPosition(self, _: float, **kwds) -> ColorDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ColorDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ColorDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> ColorDatum: ...

    @overload
    def title(self, _: str, **kwds) -> ColorDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> ColorDatum: ...

    @overload
    def title(self, _: None, **kwds) -> ColorDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> ColorDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ColorValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    ColorValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ColorValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> ColorValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Column(FieldChannelMixin, core.RowColumnEncodingFieldDef):
    r"""
    Column schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : :class:`LayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to row/column facet's subplot. The supported string values
        are ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    center : bool
        Boolean flag indicating if facet's subviews should be centered relative to their
        respective rows or columns.

        **Default value:** ``false``
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, None, :class:`Header`
        An object defining properties of a facet's header.
    sort : dict, None, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : float
        The spacing in pixels between facet's sub-views.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "column"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Column: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Column: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Column: ...

    @overload
    def align(self, _: LayoutAlign_T, **kwds) -> Column: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Column: ...

    @overload
    def bin(self, _: bool, **kwds) -> Column: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Column: ...

    @overload
    def bin(self, _: None, **kwds) -> Column: ...

    @overload
    def center(self, _: bool, **kwds) -> Column: ...

    @overload
    def field(self, _: str, **kwds) -> Column: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Column: ...

    @overload
    def header(
        self,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Column: ...

    @overload
    def header(self, _: None, **kwds) -> Column: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Column: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Column: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Column: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Column: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Column: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Column: ...

    @overload
    def sort(self, _: None, **kwds) -> Column: ...

    @overload
    def spacing(self, _: float, **kwds) -> Column: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Column: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Column: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Column: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Column: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Column: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Column: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Column: ...

    @overload
    def title(self, _: str, **kwds) -> Column: ...

    @overload
    def title(self, _: list[str], **kwds) -> Column: ...

    @overload
    def title(self, _: None, **kwds) -> Column: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Column: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        align: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        center: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        header: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | SortOrder_T
        ] = Undefined,
        spacing: Optional[float] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            center=center,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Description(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Description schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "description"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Description: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Description: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Description: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Description: ...

    @overload
    def bin(self, _: bool, **kwds) -> Description: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Description: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Description: ...

    @overload
    def bin(self, _: None, **kwds) -> Description: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Description: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Description: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], **kwds
    ) -> Description: ...

    @overload
    def field(self, _: str, **kwds) -> Description: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Description: ...

    @overload
    def format(self, _: str, **kwds) -> Description: ...

    @overload
    def format(self, _: dict, **kwds) -> Description: ...

    @overload
    def formatType(self, _: str, **kwds) -> Description: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Description: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Description: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Description: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Description: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Description: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Description: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Description: ...

    @overload
    def title(self, _: str, **kwds) -> Description: ...

    @overload
    def title(self, _: list[str], **kwds) -> Description: ...

    @overload
    def title(self, _: None, **kwds) -> Description: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Description: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class DescriptionValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    DescriptionValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "description"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> DescriptionValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> DescriptionValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Detail(FieldChannelMixin, core.FieldDefWithoutScale):
    r"""
    Detail schema wrapper.

    Definition object for a data field, its type and transformation of an encoding channel.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "detail"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Detail: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Detail: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Detail: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Detail: ...

    @overload
    def bin(self, _: bool, **kwds) -> Detail: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Detail: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Detail: ...

    @overload
    def bin(self, _: None, **kwds) -> Detail: ...

    @overload
    def field(self, _: str, **kwds) -> Detail: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Detail: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Detail: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Detail: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Detail: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Detail: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Detail: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Detail: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Detail: ...

    @overload
    def title(self, _: str, **kwds) -> Detail: ...

    @overload
    def title(self, _: list[str], **kwds) -> Detail: ...

    @overload
    def title(self, _: None, **kwds) -> Detail: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Detail: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Facet(FieldChannelMixin, core.FacetEncodingFieldDef):
    r"""
    Facet schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : dict, :class:`LayoutAlign`, :class:`RowColLayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to grid rows and columns. The supported string values are
        ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        Alternatively, an object value of the form ``{"row": string, "column": string}`` can
        be used to supply different alignments for rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    bounds : Literal['full', 'flush']
        The bounds calculation method to use for determining the extent of a sub-plot. One
        of ``full`` (the default) or ``flush``.

        * If set to ``full``, the entire calculated bounds (including axes, title, and
          legend) will be used.
        * If set to ``flush``, only the specified width and height values for the sub-view
          will be used. The ``flush`` setting can be useful when attempting to place
          sub-plots without axes or legends into a uniform grid structure.

        **Default value:** ``"full"``
    center : bool, dict, :class:`RowColboolean`
        Boolean flag indicating if subviews should be centered relative to their respective
        rows or columns.

        An object value of the form ``{"row": boolean, "column": boolean}`` can be used to
        supply different centering values for rows and columns.

        **Default value:** ``false``
    columns : float
        The number of columns to include in the view composition layout.

        **Default value**: ``undefined`` -- An infinite number of columns (a single row)
        will be assumed. This is equivalent to ``hconcat`` (for ``concat``) and to using the
        ``column`` channel (for ``facet`` and ``repeat``).

        **Note**:

        1) This property is only for:

        * the general (wrappable) ``concat`` operator (not ``hconcat``/``vconcat``)
        * the ``facet`` and ``repeat`` operator with one field/repetition definition
          (without row/column nesting)

        2) Setting the ``columns`` to ``1`` is equivalent to ``vconcat`` (for ``concat``)
        and to using the ``row`` channel (for ``facet`` and ``repeat``).
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, None, :class:`Header`
        An object defining properties of a facet's header.
    sort : dict, None, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : dict, float, :class:`RowColnumber`
        The spacing in pixels between sub-views of the composition operator. An object of
        the form ``{"row": number, "column": number}`` can be used to set different spacing
        values for rows and columns.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "facet"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Facet: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Facet: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Facet: ...

    @overload
    def align(self, _: LayoutAlign_T, **kwds) -> Facet: ...

    @overload
    def align(
        self,
        column: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        row: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Facet: ...

    @overload
    def bin(self, _: bool, **kwds) -> Facet: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def bin(self, _: None, **kwds) -> Facet: ...

    @overload
    def bounds(self, _: Literal["full", "flush"], **kwds) -> Facet: ...

    @overload
    def center(self, _: bool, **kwds) -> Facet: ...

    @overload
    def center(
        self,
        column: Optional[bool] = Undefined,
        row: Optional[bool] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def columns(self, _: float, **kwds) -> Facet: ...

    @overload
    def field(self, _: str, **kwds) -> Facet: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def header(
        self,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def header(self, _: None, **kwds) -> Facet: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Facet: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Facet: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Facet: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Facet: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Facet: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def sort(self, _: None, **kwds) -> Facet: ...

    @overload
    def spacing(self, _: float, **kwds) -> Facet: ...

    @overload
    def spacing(
        self,
        column: Optional[float] = Undefined,
        row: Optional[float] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Facet: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Facet: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Facet: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Facet: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Facet: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Facet: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Facet: ...

    @overload
    def title(self, _: str, **kwds) -> Facet: ...

    @overload
    def title(self, _: list[str], **kwds) -> Facet: ...

    @overload
    def title(self, _: None, **kwds) -> Facet: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Facet: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        align: Optional[dict | SchemaBase | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        bounds: Optional[Literal["full", "flush"]] = Undefined,
        center: Optional[bool | dict | SchemaBase] = Undefined,
        columns: Optional[float] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        header: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | SortOrder_T
        ] = Undefined,
        spacing: Optional[dict | float | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            bounds=bounds,
            center=center,
            columns=columns,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Fill(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Fill schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Fill: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Fill: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Fill: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Fill: ...

    @overload
    def bin(self, _: bool, **kwds) -> Fill: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def bin(self, _: None, **kwds) -> Fill: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> Fill: ...

    @overload
    def field(self, _: str, **kwds) -> Fill: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def legend(self, _: None, **kwds) -> Fill: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def scale(self, _: None, **kwds) -> Fill: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Fill: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Fill: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Fill: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Fill: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Fill: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Fill: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Fill: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def sort(self, _: None, **kwds) -> Fill: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Fill: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Fill: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Fill: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Fill: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Fill: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Fill: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Fill: ...

    @overload
    def title(self, _: str, **kwds) -> Fill: ...

    @overload
    def title(self, _: list[str], **kwds) -> Fill: ...

    @overload
    def title(self, _: None, **kwds) -> Fill: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Fill: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    FillDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def bandPosition(self, _: float, **kwds) -> FillDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> FillDatum: ...

    @overload
    def title(self, _: str, **kwds) -> FillDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> FillDatum: ...

    @overload
    def title(self, _: None, **kwds) -> FillDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> FillDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    FillValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> FillValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class FillOpacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    FillOpacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> FillOpacity: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> FillOpacity: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> FillOpacity: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> FillOpacity: ...

    @overload
    def bin(self, _: bool, **kwds) -> FillOpacity: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def bin(self, _: None, **kwds) -> FillOpacity: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> FillOpacity: ...

    @overload
    def field(self, _: str, **kwds) -> FillOpacity: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def legend(self, _: None, **kwds) -> FillOpacity: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def scale(self, _: None, **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: list[float], **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: list[str], **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> FillOpacity: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> FillOpacity: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def sort(self, _: None, **kwds) -> FillOpacity: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> FillOpacity: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> FillOpacity: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> FillOpacity: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> FillOpacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> FillOpacity: ...

    @overload
    def title(self, _: str, **kwds) -> FillOpacity: ...

    @overload
    def title(self, _: list[str], **kwds) -> FillOpacity: ...

    @overload
    def title(self, _: None, **kwds) -> FillOpacity: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> FillOpacity: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillOpacityDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    FillOpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def bandPosition(self, _: float, **kwds) -> FillOpacityDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacityDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacityDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> FillOpacityDatum: ...

    @overload
    def title(self, _: str, **kwds) -> FillOpacityDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> FillOpacityDatum: ...

    @overload
    def title(self, _: None, **kwds) -> FillOpacityDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> FillOpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillOpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    FillOpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> FillOpacityValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> FillOpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Href(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Href schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "href"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Href: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Href: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Href: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Href: ...

    @overload
    def bin(self, _: bool, **kwds) -> Href: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Href: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Href: ...

    @overload
    def bin(self, _: None, **kwds) -> Href: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Href: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Href: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], **kwds
    ) -> Href: ...

    @overload
    def field(self, _: str, **kwds) -> Href: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Href: ...

    @overload
    def format(self, _: str, **kwds) -> Href: ...

    @overload
    def format(self, _: dict, **kwds) -> Href: ...

    @overload
    def formatType(self, _: str, **kwds) -> Href: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Href: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Href: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Href: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Href: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Href: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Href: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Href: ...

    @overload
    def title(self, _: str, **kwds) -> Href: ...

    @overload
    def title(self, _: list[str], **kwds) -> Href: ...

    @overload
    def title(self, _: None, **kwds) -> Href: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Href: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class HrefValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    HrefValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "href"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> HrefValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> HrefValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Key(FieldChannelMixin, core.FieldDefWithoutScale):
    r"""
    Key schema wrapper.

    Definition object for a data field, its type and transformation of an encoding channel.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "key"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Key: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Key: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Key: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Key: ...

    @overload
    def bin(self, _: bool, **kwds) -> Key: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Key: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Key: ...

    @overload
    def bin(self, _: None, **kwds) -> Key: ...

    @overload
    def field(self, _: str, **kwds) -> Key: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Key: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Key: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Key: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Key: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Key: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Key: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Key: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Key: ...

    @overload
    def title(self, _: str, **kwds) -> Key: ...

    @overload
    def title(self, _: list[str], **kwds) -> Key: ...

    @overload
    def title(self, _: None, **kwds) -> Key: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Key: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Latitude(FieldChannelMixin, core.LatLongFieldDef):
    r"""
    Latitude schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : Literal['quantitative']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Latitude: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Latitude: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Latitude: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Latitude: ...

    @overload
    def bin(self, _: None, **kwds) -> Latitude: ...

    @overload
    def field(self, _: str, **kwds) -> Latitude: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Latitude: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Latitude: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Latitude: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Latitude: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Latitude: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Latitude: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Latitude: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Latitude: ...

    @overload
    def title(self, _: str, **kwds) -> Latitude: ...

    @overload
    def title(self, _: list[str], **kwds) -> Latitude: ...

    @overload
    def title(self, _: None, **kwds) -> Latitude: ...

    @overload
    def type(self, _: Literal["quantitative"], **kwds) -> Latitude: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[Literal["quantitative"]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class LatitudeDatum(DatumChannelMixin, core.DatumDef):
    """
    LatitudeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude"

    @overload
    def bandPosition(self, _: float, **kwds) -> LatitudeDatum: ...

    @overload
    def title(self, _: str, **kwds) -> LatitudeDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> LatitudeDatum: ...

    @overload
    def title(self, _: None, **kwds) -> LatitudeDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> LatitudeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Latitude2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Latitude2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Latitude2: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Latitude2: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Latitude2: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Latitude2: ...

    @overload
    def bin(self, _: None, **kwds) -> Latitude2: ...

    @overload
    def field(self, _: str, **kwds) -> Latitude2: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Latitude2: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Latitude2: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Latitude2: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Latitude2: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Latitude2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Latitude2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Latitude2: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Latitude2: ...

    @overload
    def title(self, _: str, **kwds) -> Latitude2: ...

    @overload
    def title(self, _: list[str], **kwds) -> Latitude2: ...

    @overload
    def title(self, _: None, **kwds) -> Latitude2: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Latitude2Datum(DatumChannelMixin, core.DatumDef):
    """
    Latitude2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    @overload
    def bandPosition(self, _: float, **kwds) -> Latitude2Datum: ...

    @overload
    def title(self, _: str, **kwds) -> Latitude2Datum: ...

    @overload
    def title(self, _: list[str], **kwds) -> Latitude2Datum: ...

    @overload
    def title(self, _: None, **kwds) -> Latitude2Datum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> Latitude2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Latitude2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Latitude2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Longitude(FieldChannelMixin, core.LatLongFieldDef):
    r"""
    Longitude schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : Literal['quantitative']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Longitude: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Longitude: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Longitude: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Longitude: ...

    @overload
    def bin(self, _: None, **kwds) -> Longitude: ...

    @overload
    def field(self, _: str, **kwds) -> Longitude: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Longitude: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Longitude: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Longitude: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Longitude: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Longitude: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Longitude: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Longitude: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Longitude: ...

    @overload
    def title(self, _: str, **kwds) -> Longitude: ...

    @overload
    def title(self, _: list[str], **kwds) -> Longitude: ...

    @overload
    def title(self, _: None, **kwds) -> Longitude: ...

    @overload
    def type(self, _: Literal["quantitative"], **kwds) -> Longitude: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[Literal["quantitative"]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class LongitudeDatum(DatumChannelMixin, core.DatumDef):
    """
    LongitudeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude"

    @overload
    def bandPosition(self, _: float, **kwds) -> LongitudeDatum: ...

    @overload
    def title(self, _: str, **kwds) -> LongitudeDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> LongitudeDatum: ...

    @overload
    def title(self, _: None, **kwds) -> LongitudeDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> LongitudeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Longitude2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Longitude2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Longitude2: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Longitude2: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Longitude2: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Longitude2: ...

    @overload
    def bin(self, _: None, **kwds) -> Longitude2: ...

    @overload
    def field(self, _: str, **kwds) -> Longitude2: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Longitude2: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Longitude2: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Longitude2: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Longitude2: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Longitude2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Longitude2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Longitude2: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Longitude2: ...

    @overload
    def title(self, _: str, **kwds) -> Longitude2: ...

    @overload
    def title(self, _: list[str], **kwds) -> Longitude2: ...

    @overload
    def title(self, _: None, **kwds) -> Longitude2: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Longitude2Datum(DatumChannelMixin, core.DatumDef):
    """
    Longitude2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    @overload
    def bandPosition(self, _: float, **kwds) -> Longitude2Datum: ...

    @overload
    def title(self, _: str, **kwds) -> Longitude2Datum: ...

    @overload
    def title(self, _: list[str], **kwds) -> Longitude2Datum: ...

    @overload
    def title(self, _: None, **kwds) -> Longitude2Datum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> Longitude2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Longitude2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Longitude2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Opacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    Opacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Opacity: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Opacity: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Opacity: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Opacity: ...

    @overload
    def bin(self, _: bool, **kwds) -> Opacity: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def bin(self, _: None, **kwds) -> Opacity: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> Opacity: ...

    @overload
    def field(self, _: str, **kwds) -> Opacity: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def legend(self, _: None, **kwds) -> Opacity: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def scale(self, _: None, **kwds) -> Opacity: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Opacity: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Opacity: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Opacity: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Opacity: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Opacity: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Opacity: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Opacity: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def sort(self, _: None, **kwds) -> Opacity: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Opacity: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Opacity: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Opacity: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Opacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Opacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Opacity: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Opacity: ...

    @overload
    def title(self, _: str, **kwds) -> Opacity: ...

    @overload
    def title(self, _: list[str], **kwds) -> Opacity: ...

    @overload
    def title(self, _: None, **kwds) -> Opacity: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Opacity: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OpacityDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    OpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def bandPosition(self, _: float, **kwds) -> OpacityDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> OpacityDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> OpacityDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> OpacityDatum: ...

    @overload
    def title(self, _: str, **kwds) -> OpacityDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> OpacityDatum: ...

    @overload
    def title(self, _: None, **kwds) -> OpacityDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> OpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    OpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> OpacityValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> OpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Order(FieldChannelMixin, core.OrderFieldDef):
    r"""
    Order schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    sort : :class:`SortOrder`, Literal['ascending', 'descending']
        The sort order. One of ``"ascending"`` (default) or ``"descending"``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "order"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Order: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Order: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Order: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Order: ...

    @overload
    def bin(self, _: bool, **kwds) -> Order: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Order: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Order: ...

    @overload
    def bin(self, _: None, **kwds) -> Order: ...

    @overload
    def field(self, _: str, **kwds) -> Order: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Order: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Order: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Order: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Order: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Order: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Order: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Order: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Order: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Order: ...

    @overload
    def title(self, _: str, **kwds) -> Order: ...

    @overload
    def title(self, _: list[str], **kwds) -> Order: ...

    @overload
    def title(self, _: None, **kwds) -> Order: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Order: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        sort: Optional[SchemaBase | SortOrder_T] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OrderValue(ValueChannelMixin, core.OrderValueDef):
    """
    OrderValue schema wrapper.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    condition : dict, :class:`ConditionalValueDefnumber`, :class:`ConditionalParameterValueDefnumber`, :class:`ConditionalPredicateValueDefnumber`, Sequence[dict, :class:`ConditionalValueDefnumber`, :class:`ConditionalParameterValueDefnumber`, :class:`ConditionalPredicateValueDefnumber`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "order"

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[float] = Undefined,
        **kwds,
    ) -> OrderValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float] = Undefined,
        **kwds,
    ) -> OrderValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumber], **kwds
    ) -> OrderValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Radius(FieldChannelMixin, core.PositionFieldDefBase):
    r"""
    Radius schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, None, :class:`StackOffset`, Literal['zero', 'center', 'normalize']
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).  :raw-html:`<br/>`
          -``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Radius: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Radius: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Radius: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Radius: ...

    @overload
    def bin(self, _: bool, **kwds) -> Radius: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Radius: ...

    @overload
    def bin(self, _: None, **kwds) -> Radius: ...

    @overload
    def field(self, _: str, **kwds) -> Radius: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def scale(self, _: None, **kwds) -> Radius: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Radius: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Radius: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Radius: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Radius: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Radius: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Radius: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Radius: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def sort(self, _: None, **kwds) -> Radius: ...

    @overload
    def stack(self, _: StackOffset_T, **kwds) -> Radius: ...

    @overload
    def stack(self, _: None, **kwds) -> Radius: ...

    @overload
    def stack(self, _: bool, **kwds) -> Radius: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Radius: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Radius: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Radius: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Radius: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Radius: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Radius: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Radius: ...

    @overload
    def title(self, _: str, **kwds) -> Radius: ...

    @overload
    def title(self, _: list[str], **kwds) -> Radius: ...

    @overload
    def title(self, _: None, **kwds) -> Radius: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Radius: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        stack: Optional[bool | None | SchemaBase | StackOffset_T] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class RadiusDatum(DatumChannelMixin, core.PositionDatumDefBase):
    """
    RadiusDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    stack : bool, None, :class:`StackOffset`, Literal['zero', 'center', 'normalize']
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).  :raw-html:`<br/>`
          -``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    @overload
    def bandPosition(self, _: float, **kwds) -> RadiusDatum: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> RadiusDatum: ...

    @overload
    def scale(self, _: None, **kwds) -> RadiusDatum: ...

    @overload
    def stack(self, _: StackOffset_T, **kwds) -> RadiusDatum: ...

    @overload
    def stack(self, _: None, **kwds) -> RadiusDatum: ...

    @overload
    def stack(self, _: bool, **kwds) -> RadiusDatum: ...

    @overload
    def title(self, _: str, **kwds) -> RadiusDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> RadiusDatum: ...

    @overload
    def title(self, _: None, **kwds) -> RadiusDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> RadiusDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        stack: Optional[bool | None | SchemaBase | StackOffset_T] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            scale=scale,
            stack=stack,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class RadiusValue(ValueChannelMixin, core.PositionValueDef):
    """
    RadiusValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Radius2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Radius2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Radius2: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Radius2: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Radius2: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Radius2: ...

    @overload
    def bin(self, _: None, **kwds) -> Radius2: ...

    @overload
    def field(self, _: str, **kwds) -> Radius2: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Radius2: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Radius2: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Radius2: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Radius2: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Radius2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Radius2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Radius2: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Radius2: ...

    @overload
    def title(self, _: str, **kwds) -> Radius2: ...

    @overload
    def title(self, _: list[str], **kwds) -> Radius2: ...

    @overload
    def title(self, _: None, **kwds) -> Radius2: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Radius2Datum(DatumChannelMixin, core.DatumDef):
    """
    Radius2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    @overload
    def bandPosition(self, _: float, **kwds) -> Radius2Datum: ...

    @overload
    def title(self, _: str, **kwds) -> Radius2Datum: ...

    @overload
    def title(self, _: list[str], **kwds) -> Radius2Datum: ...

    @overload
    def title(self, _: None, **kwds) -> Radius2Datum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> Radius2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Radius2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Radius2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Row(FieldChannelMixin, core.RowColumnEncodingFieldDef):
    r"""
    Row schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : :class:`LayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to row/column facet's subplot. The supported string values
        are ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    center : bool
        Boolean flag indicating if facet's subviews should be centered relative to their
        respective rows or columns.

        **Default value:** ``false``
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, None, :class:`Header`
        An object defining properties of a facet's header.
    sort : dict, None, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : float
        The spacing in pixels between facet's sub-views.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "row"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Row: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Row: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Row: ...

    @overload
    def align(self, _: LayoutAlign_T, **kwds) -> Row: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Row: ...

    @overload
    def bin(self, _: bool, **kwds) -> Row: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Row: ...

    @overload
    def bin(self, _: None, **kwds) -> Row: ...

    @overload
    def center(self, _: bool, **kwds) -> Row: ...

    @overload
    def field(self, _: str, **kwds) -> Row: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Row: ...

    @overload
    def header(
        self,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Row: ...

    @overload
    def header(self, _: None, **kwds) -> Row: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Row: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Row: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Row: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Row: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Row: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Row: ...

    @overload
    def sort(self, _: None, **kwds) -> Row: ...

    @overload
    def spacing(self, _: float, **kwds) -> Row: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Row: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Row: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Row: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Row: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Row: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Row: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Row: ...

    @overload
    def title(self, _: str, **kwds) -> Row: ...

    @overload
    def title(self, _: list[str], **kwds) -> Row: ...

    @overload
    def title(self, _: None, **kwds) -> Row: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Row: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        align: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        center: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        header: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | SortOrder_T
        ] = Undefined,
        spacing: Optional[float] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            center=center,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Shape(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefTypeForShapestringnull,
):
    r"""
    Shape schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`TypeForShape`, Literal['nominal', 'ordinal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Shape: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Shape: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Shape: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Shape: ...

    @overload
    def bin(self, _: bool, **kwds) -> Shape: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def bin(self, _: None, **kwds) -> Shape: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> Shape: ...

    @overload
    def field(self, _: str, **kwds) -> Shape: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def legend(self, _: None, **kwds) -> Shape: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def scale(self, _: None, **kwds) -> Shape: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Shape: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Shape: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Shape: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Shape: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Shape: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Shape: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Shape: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def sort(self, _: None, **kwds) -> Shape: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Shape: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Shape: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Shape: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Shape: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Shape: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Shape: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Shape: ...

    @overload
    def title(self, _: str, **kwds) -> Shape: ...

    @overload
    def title(self, _: list[str], **kwds) -> Shape: ...

    @overload
    def title(self, _: None, **kwds) -> Shape: ...

    @overload
    def type(self, _: TypeForShape_T, **kwds) -> Shape: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ShapeDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefstringnull
):
    """
    ShapeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def bandPosition(self, _: float, **kwds) -> ShapeDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ShapeDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ShapeDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> ShapeDatum: ...

    @overload
    def title(self, _: str, **kwds) -> ShapeDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> ShapeDatum: ...

    @overload
    def title(self, _: None, **kwds) -> ShapeDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> ShapeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ShapeValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefTypeForShapestringnull,
):
    """
    ShapeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, :class:`ConditionalMarkPropFieldOrDatumDefTypeForShape`, :class:`ConditionalParameterMarkPropFieldOrDatumDefTypeForShape`, :class:`ConditionalPredicateMarkPropFieldOrDatumDefTypeForShape`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ShapeValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> ShapeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Size(FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber):
    r"""
    Size schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Size: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Size: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Size: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Size: ...

    @overload
    def bin(self, _: bool, **kwds) -> Size: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def bin(self, _: None, **kwds) -> Size: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> Size: ...

    @overload
    def field(self, _: str, **kwds) -> Size: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def legend(self, _: None, **kwds) -> Size: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def scale(self, _: None, **kwds) -> Size: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Size: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Size: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Size: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Size: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Size: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Size: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Size: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def sort(self, _: None, **kwds) -> Size: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Size: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Size: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Size: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Size: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Size: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Size: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Size: ...

    @overload
    def title(self, _: str, **kwds) -> Size: ...

    @overload
    def title(self, _: list[str], **kwds) -> Size: ...

    @overload
    def title(self, _: None, **kwds) -> Size: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Size: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class SizeDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    SizeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def bandPosition(self, _: float, **kwds) -> SizeDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> SizeDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> SizeDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> SizeDatum: ...

    @overload
    def title(self, _: str, **kwds) -> SizeDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> SizeDatum: ...

    @overload
    def title(self, _: None, **kwds) -> SizeDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> SizeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class SizeValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    SizeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> SizeValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> SizeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Stroke(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Stroke schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Stroke: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Stroke: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Stroke: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Stroke: ...

    @overload
    def bin(self, _: bool, **kwds) -> Stroke: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def bin(self, _: None, **kwds) -> Stroke: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> Stroke: ...

    @overload
    def field(self, _: str, **kwds) -> Stroke: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def legend(self, _: None, **kwds) -> Stroke: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def scale(self, _: None, **kwds) -> Stroke: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Stroke: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Stroke: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Stroke: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Stroke: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Stroke: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Stroke: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Stroke: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def sort(self, _: None, **kwds) -> Stroke: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Stroke: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Stroke: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Stroke: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Stroke: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Stroke: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Stroke: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Stroke: ...

    @overload
    def title(self, _: str, **kwds) -> Stroke: ...

    @overload
    def title(self, _: list[str], **kwds) -> Stroke: ...

    @overload
    def title(self, _: None, **kwds) -> Stroke: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Stroke: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    StrokeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> StrokeDatum: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeDatum: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> StrokeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    StrokeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], **kwds
    ) -> StrokeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeDash(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumberArray
):
    r"""
    StrokeDash schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> StrokeDash: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeDash: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeDash: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeDash: ...

    @overload
    def bin(self, _: bool, **kwds) -> StrokeDash: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def bin(self, _: None, **kwds) -> StrokeDash: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], **kwds
    ) -> StrokeDash: ...

    @overload
    def field(self, _: str, **kwds) -> StrokeDash: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def legend(self, _: None, **kwds) -> StrokeDash: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def scale(self, _: None, **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: list[float], **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: list[str], **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> StrokeDash: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> StrokeDash: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def sort(self, _: None, **kwds) -> StrokeDash: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> StrokeDash: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> StrokeDash: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> StrokeDash: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> StrokeDash: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> StrokeDash: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeDash: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeDash: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeDash: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> StrokeDash: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDashDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumberArray
):
    """
    StrokeDashDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeDashDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDashDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDashDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], **kwds
    ) -> StrokeDashDatum: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeDashDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeDashDatum: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeDashDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> StrokeDashDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDashValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumberArray
):
    """
    StrokeDashValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, Sequence[float], :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeDashValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], **kwds
    ) -> StrokeDashValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeOpacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    StrokeOpacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> StrokeOpacity: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeOpacity: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeOpacity: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeOpacity: ...

    @overload
    def bin(self, _: bool, **kwds) -> StrokeOpacity: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def bin(self, _: None, **kwds) -> StrokeOpacity: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeOpacity: ...

    @overload
    def field(self, _: str, **kwds) -> StrokeOpacity: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def legend(self, _: None, **kwds) -> StrokeOpacity: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def scale(self, _: None, **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: list[float], **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: list[str], **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> StrokeOpacity: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> StrokeOpacity: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def sort(self, _: None, **kwds) -> StrokeOpacity: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> StrokeOpacity: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> StrokeOpacity: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> StrokeOpacity: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> StrokeOpacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> StrokeOpacity: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeOpacity: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeOpacity: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeOpacity: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> StrokeOpacity: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeOpacityDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    StrokeOpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeOpacityDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacityDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacityDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeOpacityDatum: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeOpacityDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeOpacityDatum: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeOpacityDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> StrokeOpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeOpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    StrokeOpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeOpacityValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeOpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeWidth(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    StrokeWidth schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, None, :class:`Legend`
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> StrokeWidth: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeWidth: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> StrokeWidth: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeWidth: ...

    @overload
    def bin(self, _: bool, **kwds) -> StrokeWidth: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def bin(self, _: None, **kwds) -> StrokeWidth: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeWidth: ...

    @overload
    def field(self, _: str, **kwds) -> StrokeWidth: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def legend(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        clipHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columnPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        columns: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        cornerRadius: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gradientStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gradientStrokeWidth: Optional[
            dict | float | Parameter | SchemaBase
        ] = Undefined,
        gradientThickness: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridAlign: Optional[dict | Parameter | SchemaBase | LayoutAlign_T] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        legendY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        rowPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        strokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        symbolDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolFillColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolStrokeColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        symbolStrokeWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        symbolType: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOrient: Optional[dict | Parameter | SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def legend(self, _: None, **kwds) -> StrokeWidth: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def scale(self, _: None, **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: list[float], **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: list[str], **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> StrokeWidth: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> StrokeWidth: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def sort(self, _: None, **kwds) -> StrokeWidth: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> StrokeWidth: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> StrokeWidth: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> StrokeWidth: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> StrokeWidth: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> StrokeWidth: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeWidth: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeWidth: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeWidth: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> StrokeWidth: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeWidthDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    StrokeWidthDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def bandPosition(self, _: float, **kwds) -> StrokeWidthDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidthDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidthDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeWidthDatum: ...

    @overload
    def title(self, _: str, **kwds) -> StrokeWidthDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> StrokeWidthDatum: ...

    @overload
    def title(self, _: None, **kwds) -> StrokeWidthDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> StrokeWidthDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeWidthValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    StrokeWidthValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> StrokeWidthValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], **kwds
    ) -> StrokeWidthValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Text(FieldChannelMixin, core.FieldOrDatumDefWithConditionStringFieldDefText):
    r"""
    Text schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Text: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Text: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Text: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Text: ...

    @overload
    def bin(self, _: bool, **kwds) -> Text: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Text: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Text: ...

    @overload
    def bin(self, _: None, **kwds) -> Text: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> Text: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> Text: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefTextExprRef], **kwds
    ) -> Text: ...

    @overload
    def field(self, _: str, **kwds) -> Text: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Text: ...

    @overload
    def format(self, _: str, **kwds) -> Text: ...

    @overload
    def format(self, _: dict, **kwds) -> Text: ...

    @overload
    def formatType(self, _: str, **kwds) -> Text: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Text: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Text: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Text: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Text: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Text: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Text: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Text: ...

    @overload
    def title(self, _: str, **kwds) -> Text: ...

    @overload
    def title(self, _: list[str], **kwds) -> Text: ...

    @overload
    def title(self, _: None, **kwds) -> Text: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Text: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TextDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionStringDatumDefText):
    """
    TextDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def bandPosition(self, _: float, **kwds) -> TextDatum: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> TextDatum: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> TextDatum: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefTextExprRef], **kwds
    ) -> TextDatum: ...

    @overload
    def format(self, _: str, **kwds) -> TextDatum: ...

    @overload
    def format(self, _: dict, **kwds) -> TextDatum: ...

    @overload
    def formatType(self, _: str, **kwds) -> TextDatum: ...

    @overload
    def title(self, _: str, **kwds) -> TextDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> TextDatum: ...

    @overload
    def title(self, _: None, **kwds) -> TextDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> TextDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            format=format,
            formatType=formatType,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TextValue(ValueChannelMixin, core.ValueDefWithConditionStringFieldDefText):
    """
    TextValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalStringFieldDef`, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterStringFieldDef`, :class:`ConditionalPredicateStringFieldDef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`Text`, Sequence[str], :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> TextValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> TextValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> TextValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[
            str | dict | Parameter | SchemaBase | Sequence[str]
        ] = Undefined,
        **kwds,
    ) -> TextValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefTextExprRef], **kwds
    ) -> TextValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Theta(FieldChannelMixin, core.PositionFieldDefBase):
    r"""
    Theta schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, None, :class:`StackOffset`, Literal['zero', 'center', 'normalize']
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).  :raw-html:`<br/>`
          -``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Theta: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Theta: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Theta: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Theta: ...

    @overload
    def bin(self, _: bool, **kwds) -> Theta: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Theta: ...

    @overload
    def bin(self, _: None, **kwds) -> Theta: ...

    @overload
    def field(self, _: str, **kwds) -> Theta: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def scale(self, _: None, **kwds) -> Theta: ...

    @overload
    def sort(self, _: list[float], **kwds) -> Theta: ...

    @overload
    def sort(self, _: list[str], **kwds) -> Theta: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> Theta: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> Theta: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> Theta: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> Theta: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> Theta: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def sort(self, _: None, **kwds) -> Theta: ...

    @overload
    def stack(self, _: StackOffset_T, **kwds) -> Theta: ...

    @overload
    def stack(self, _: None, **kwds) -> Theta: ...

    @overload
    def stack(self, _: bool, **kwds) -> Theta: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Theta: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Theta: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Theta: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Theta: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Theta: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Theta: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Theta: ...

    @overload
    def title(self, _: str, **kwds) -> Theta: ...

    @overload
    def title(self, _: list[str], **kwds) -> Theta: ...

    @overload
    def title(self, _: None, **kwds) -> Theta: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Theta: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        stack: Optional[bool | None | SchemaBase | StackOffset_T] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ThetaDatum(DatumChannelMixin, core.PositionDatumDefBase):
    """
    ThetaDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    stack : bool, None, :class:`StackOffset`, Literal['zero', 'center', 'normalize']
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).  :raw-html:`<br/>`
          -``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    @overload
    def bandPosition(self, _: float, **kwds) -> ThetaDatum: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> ThetaDatum: ...

    @overload
    def scale(self, _: None, **kwds) -> ThetaDatum: ...

    @overload
    def stack(self, _: StackOffset_T, **kwds) -> ThetaDatum: ...

    @overload
    def stack(self, _: None, **kwds) -> ThetaDatum: ...

    @overload
    def stack(self, _: bool, **kwds) -> ThetaDatum: ...

    @overload
    def title(self, _: str, **kwds) -> ThetaDatum: ...

    @overload
    def title(self, _: list[str], **kwds) -> ThetaDatum: ...

    @overload
    def title(self, _: None, **kwds) -> ThetaDatum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> ThetaDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        stack: Optional[bool | None | SchemaBase | StackOffset_T] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            scale=scale,
            stack=stack,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ThetaValue(ValueChannelMixin, core.PositionValueDef):
    """
    ThetaValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Theta2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Theta2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Theta2: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Theta2: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Theta2: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Theta2: ...

    @overload
    def bin(self, _: None, **kwds) -> Theta2: ...

    @overload
    def field(self, _: str, **kwds) -> Theta2: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Theta2: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Theta2: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Theta2: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Theta2: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Theta2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Theta2: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Theta2: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Theta2: ...

    @overload
    def title(self, _: str, **kwds) -> Theta2: ...

    @overload
    def title(self, _: list[str], **kwds) -> Theta2: ...

    @overload
    def title(self, _: None, **kwds) -> Theta2: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Theta2Datum(DatumChannelMixin, core.DatumDef):
    """
    Theta2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    @overload
    def bandPosition(self, _: float, **kwds) -> Theta2Datum: ...

    @overload
    def title(self, _: str, **kwds) -> Theta2Datum: ...

    @overload
    def title(self, _: list[str], **kwds) -> Theta2Datum: ...

    @overload
    def title(self, _: None, **kwds) -> Theta2Datum: ...

    @overload
    def type(self, _: Type_T, **kwds) -> Theta2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Theta2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Theta2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Tooltip(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Tooltip schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "tooltip"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Tooltip: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Tooltip: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Tooltip: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Tooltip: ...

    @overload
    def bin(self, _: bool, **kwds) -> Tooltip: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Tooltip: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Tooltip: ...

    @overload
    def bin(self, _: None, **kwds) -> Tooltip: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Tooltip: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Tooltip: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], **kwds
    ) -> Tooltip: ...

    @overload
    def field(self, _: str, **kwds) -> Tooltip: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Tooltip: ...

    @overload
    def format(self, _: str, **kwds) -> Tooltip: ...

    @overload
    def format(self, _: dict, **kwds) -> Tooltip: ...

    @overload
    def formatType(self, _: str, **kwds) -> Tooltip: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Tooltip: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Tooltip: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Tooltip: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Tooltip: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Tooltip: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Tooltip: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Tooltip: ...

    @overload
    def title(self, _: str, **kwds) -> Tooltip: ...

    @overload
    def title(self, _: list[str], **kwds) -> Tooltip: ...

    @overload
    def title(self, _: None, **kwds) -> Tooltip: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Tooltip: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TooltipValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    TooltipValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "tooltip"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> TooltipValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> TooltipValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Url(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Url schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "url"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> Url: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Url: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> Url: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> Url: ...

    @overload
    def bin(self, _: bool, **kwds) -> Url: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> Url: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> Url: ...

    @overload
    def bin(self, _: None, **kwds) -> Url: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Url: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> Url: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], **kwds
    ) -> Url: ...

    @overload
    def field(self, _: str, **kwds) -> Url: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> Url: ...

    @overload
    def format(self, _: str, **kwds) -> Url: ...

    @overload
    def format(self, _: dict, **kwds) -> Url: ...

    @overload
    def formatType(self, _: str, **kwds) -> Url: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> Url: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> Url: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> Url: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> Url: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> Url: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> Url: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> Url: ...

    @overload
    def title(self, _: str, **kwds) -> Url: ...

    @overload
    def title(self, _: list[str], **kwds) -> Url: ...

    @overload
    def title(self, _: None, **kwds) -> Url: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> Url: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class UrlValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    UrlValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, None, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "url"

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        test: Optional[str | dict | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            str | bool | dict | None | float | Parameter | SchemaBase
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[dict | None | SchemaBase] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self,
        test: Optional[str | dict | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | dict | None | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> UrlValue: ...

    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], **kwds
    ) -> UrlValue: ...

    def __init__(
        self,
        value,
        condition: Optional[
            dict | SchemaBase | Sequence[dict | SchemaBase]
        ] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class X(FieldChannelMixin, core.PositionFieldDef):
    r"""
    X schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    axis : dict, None, :class:`Axis`
        An object defining properties of axis's gridlines, ticks and labels. If ``null``,
        the axis for the encoding channel will be removed.

        **Default value:** If undefined, default `axis properties
        <https://vega.github.io/vega-lite/docs/axis.html>`__ are applied.

        **See also:** `axis <https://vega.github.io/vega-lite/docs/axis.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, None, Literal['binned'], :class:`BinParams`
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    impute : dict, None, :class:`ImputeParams`
        An object defining the properties of the Impute Operation to be applied. The field
        value of the other positional channel is taken as ``key`` of the ``Impute``
        Operation. The field of the ``color`` channel if specified is used as ``groupby`` of
        the ``Impute`` Operation.

        **See also:** `impute <https://vega.github.io/vega-lite/docs/impute.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, None, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text']
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, None, :class:`StackOffset`, Literal['zero', 'center', 'normalize']
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).  :raw-html:`<br/>`
          -``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, None, :class:`Text`, Sequence[str]
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, **kwds) -> X: ...

    @overload
    def aggregate(
        self, argmax: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> X: ...

    @overload
    def aggregate(
        self, argmin: Optional[str | SchemaBase] = Undefined, **kwds
    ) -> X: ...

    @overload
    def axis(
        self,
        aria: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        bandPosition: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        description: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        domain: Optional[bool] = Undefined,
        domainCap: Optional[dict | Parameter | SchemaBase | StrokeCap_T] = Undefined,
        domainColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        domainDash: Optional[
            dict | Parameter | SchemaBase | Sequence[float]
        ] = Undefined,
        domainDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        format: Optional[str | dict | SchemaBase] = Undefined,
        formatType: Optional[str] = Undefined,
        grid: Optional[bool] = Undefined,
        gridCap: Optional[dict | Parameter | SchemaBase | StrokeCap_T] = Undefined,
        gridColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        gridDash: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        gridDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        gridWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        labelAngle: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        labelBound: Optional[bool | dict | float | Parameter | SchemaBase] = Undefined,
        labelColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFlush: Optional[bool | float] = Undefined,
        labelFlushOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        labelFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelOverlap: Optional[
            bool | dict | Parameter | SchemaBase | Literal["greedy", "parity"]
        ] = Undefined,
        labelPadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labelSeparation: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        labels: Optional[bool] = Undefined,
        maxExtent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        minExtent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        offset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        orient: Optional[dict | Parameter | SchemaBase | AxisOrient_T] = Undefined,
        position: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        style: Optional[str | Sequence[str]] = Undefined,
        tickBand: Optional[
            dict | Parameter | SchemaBase | Literal["center", "extent"]
        ] = Undefined,
        tickCap: Optional[dict | Parameter | SchemaBase | StrokeCap_T] = Undefined,
        tickColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        tickCount: Optional[
            dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        tickDash: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        tickDashOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        tickExtra: Optional[bool] = Undefined,
        tickMinStep: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        tickOffset: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        tickOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        tickRound: Optional[bool] = Undefined,
        tickSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        tickWidth: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        ticks: Optional[bool] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        titleAlign: Optional[dict | Parameter | SchemaBase | Align_T] = Undefined,
        titleAnchor: Optional[
            dict | Parameter | SchemaBase | TitleAnchor_T
        ] = Undefined,
        titleAngle: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleBaseline: Optional[
            dict | Parameter | SchemaBase | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | dict | None | Parameter | SchemaBase | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontSize: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleFontStyle: Optional[str | dict | Parameter | SchemaBase] = Undefined,
        titleFontWeight: Optional[
            dict | Parameter | SchemaBase | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleLineHeight: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleOpacity: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titlePadding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleX: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        titleY: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        translate: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        values: Optional[
            dict
            | Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
        ] = Undefined,
        zindex: Optional[float] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def axis(self, _: None, **kwds) -> X: ...

    @overload
    def bandPosition(self, _: float, **kwds) -> X: ...

    @overload
    def bin(self, _: bool, **kwds) -> X: ...

    @overload
    def bin(
        self,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[dict | Parameter | SchemaBase | Sequence[float]] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def bin(self, _: Literal["binned"], **kwds) -> X: ...

    @overload
    def bin(self, _: None, **kwds) -> X: ...

    @overload
    def field(self, _: str, **kwds) -> X: ...

    @overload
    def field(
        self,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def impute(
        self,
        frame: Optional[Sequence[None | float]] = Undefined,
        keyvals: Optional[dict | SchemaBase | Sequence[Any]] = Undefined,
        method: Optional[SchemaBase | ImputeMethod_T] = Undefined,
        value: Optional[Any] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def impute(self, _: None, **kwds) -> X: ...

    @overload
    def scale(
        self,
        align: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        base: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        bins: Optional[dict | SchemaBase | Sequence[float]] = Undefined,
        clamp: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        constant: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domain: Optional[
            dict
            | Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[str | bool | dict | None | float | Parameter | SchemaBase]
        ] = Undefined,
        domainMax: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMid: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainMin: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        domainRaw: Optional[dict | Parameter | SchemaBase] = Undefined,
        exponent: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        interpolate: Optional[
            dict | Parameter | SchemaBase | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | dict | float | Parameter | SchemaBase | TimeInterval_T
        ] = Undefined,
        padding: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingInner: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        paddingOuter: Optional[dict | float | Parameter | SchemaBase] = Undefined,
        range: Optional[
            dict
            | SchemaBase
            | Sequence[str | dict | float | Parameter | SchemaBase | Sequence[float]]
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        rangeMin: Optional[str | dict | float | Parameter | SchemaBase] = Undefined,
        reverse: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        round: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        scheme: Optional[dict | Parameter | SchemaBase | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | dict | Parameter | SchemaBase] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def scale(self, _: None, **kwds) -> X: ...

    @overload
    def sort(self, _: list[float], **kwds) -> X: ...

    @overload
    def sort(self, _: list[str], **kwds) -> X: ...

    @overload
    def sort(self, _: list[bool], **kwds) -> X: ...

    @overload
    def sort(self, _: list[core.DateTime], **kwds) -> X: ...

    @overload
    def sort(self, _: SortOrder_T, **kwds) -> X: ...

    @overload
    def sort(self, _: SortByChannel_T, **kwds) -> X: ...

    @overload
    def sort(self, _: SortByChannelDesc_T, **kwds) -> X: ...

    @overload
    def sort(
        self,
        field: Optional[str | dict | SchemaBase] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def sort(
        self,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[None | SchemaBase | SortOrder_T] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def sort(self, _: None, **kwds) -> X: ...

    @overload
    def stack(self, _: StackOffset_T, **kwds) -> X: ...

    @overload
    def stack(self, _: None, **kwds) -> X: ...

    @overload
    def stack(self, _: bool, **kwds) -> X: ...

    @overload
    def timeUnit(self, _: LocalSingleTimeUnit_T, **kwds) -> X: ...

    @overload
    def timeUnit(self, _: UtcSingleTimeUnit_T, **kwds) -> X: ...

    @overload
    def timeUnit(self, _: LocalMultiTimeUnit_T, **kwds) -> X: ...

    @overload
    def timeUnit(self, _: UtcMultiTimeUnit_T, **kwds) -> X: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedyear",
            "binnedyearquarter",
            "binnedyearquartermonth",
            "binnedyearmonth",
            "binnedyearmonthdate",
            "binnedyearmonthdatehours",
            "binnedyearmonthdatehoursminutes",
            "binnedyearmonthdatehoursminutesseconds",
            "binnedyearweek",
            "binnedyearweekday",
            "binnedyearweekdayhours",
            "binnedyearweekdayhoursminutes",
            "binnedyearweekdayhoursminutesseconds",
            "binnedyeardayofyear",
        ],
        **kwds,
    ) -> X: ...

    @overload
    def timeUnit(
        self,
        _: Literal[
            "binnedutcyear",
            "binnedutcyearquarter",
            "binnedutcyearquartermonth",
            "binnedutcyearmonth",
            "binnedutcyearmonthdate",
            "binnedutcyearmonthdatehours",
            "binnedutcyearmonthdatehoursminutes",
            "binnedutcyearmonthdatehoursminutesseconds",
            "binnedutcyearweek",
            "binnedutcyearweekday",
            "binnedutcyearweekdayhours",
            "binnedutcyearweekdayhoursminutes",
            "binnedutcyearweekdayhoursminutesseconds",
            "binnedutcyeardayofyear",
        ],
        **kwds,
    ) -> X: ...

    @overload
    def timeUnit(
        self,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
        **kwds,
    ) -> X: ...

    @overload
    def title(self, _: str, **kwds) -> X: ...

    @overload
    def title(self, _: list[str], **kwds) -> X: ...

    @overload
    def title(self, _: None, **kwds) -> X: ...

    @overload
    def type(self, _: StandardType_T, **kwds) -> X: ...

    def __init__(
        self,
        shorthand: Optional[str | dict | SchemaBase | Sequence[str]] = Undefined,
        aggregate: Optional[dict | SchemaBase | NonArgAggregateOp_T] = Undefined,
        axis: Optional[dict | None | SchemaBase] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | dict | None | SchemaBase | Literal["binned"]] = Undefined,
        field: Optional[str | dict | SchemaBase] = Undefined,
        impute: Optional[dict | None | SchemaBase] = Undefined,
        scale: Optional[dict | None | SchemaBase] = Undefined,
        sort: Optional[
            dict
            | None
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[dict | SchemaBase]
            | AllSortString_T
        ] = Undefined,
        stack: Optional[bool | None | SchemaBase | StackOffset_T] = Undefined,
        timeUnit: Optional[
            dict | SchemaBase | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | None | SchemaBase | Sequence[str]] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            axis=axis,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            impute=impute,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class XDatum(DatumChannelMixin, core.PositionDatumDef):
    """
    XDatum schema wrapper.

    Parameters
    ----------
    axis : dict, None, :class:`Axis`
        An object defining properties of axis's gridlines, ticks and labels. If ``null``,
        the axis for the encoding channel will be removed.

        **Default value:** If undefined, default `axis properties
        <https://vega.github.io/vega-lite/docs/axis.html>`__ are applied.

        **See also:** `axis <https://vega.github.io/vega-lite/docs/axis.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, None, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`
        A constant value in data domain.
    impute : dict, None, :class:`ImputeParams`
        An object defining the properties of the Impute Operation to be applied. The field
        value of the other positional channel is taken as ``key`` of the ``Impute``
        Operation. The field of the ``color`` channel if specified is used as ``groupby`` of
        the ``Impute`` Operation.

        **See also:** `impute <https://vega.github.io/vega-lite/docs/impute.html>`__
        documentation.
    scale : dict, None, :class:`Scale`
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, 